<?php
/**
 * ملف إعدادات قاعدة البيانات - CarBook
 * يحتوي على إعدادات الاتصال بقاعدة البيانات والوظائف المساعدة
 */

// منع الوصول المباشر للملف
defined('DB_ACCESS') or define('DB_ACCESS', true);

// إعدادات قاعدة البيانات
define('DB_HOST', 'localhost');
define('DB_NAME', 'carbook_db');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_CHARSET', 'utf8mb4');

// إعدادات عامة
define('SITE_URL', 'http://localhost/carbook');
define('SITE_NAME', 'CarBook');
define('ADMIN_EMAIL', 'admin@carbook.com');

// فئة الاتصال بقاعدة البيانات
class Database {
    private static $instance = null;
    private $connection;
    private $stmt;
    
    /**
     * Constructor - إنشاء اتصال بقاعدة البيانات
     */
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
            ];
            
            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch(PDOException $e) {
            die("فشل الاتصال بقاعدة البيانات: " . $e->getMessage());
        }
    }
    
    /**
     * الحصول على نسخة واحدة من قاعدة البيانات (Singleton Pattern)
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * الحصول على اتصال قاعدة البيانات
     */
    public function getConnection() {
        return $this->connection;
    }
    
    /**
     * تحضير استعلام SQL
     */
    public function prepare($sql) {
        $this->stmt = $this->connection->prepare($sql);
        return $this;
    }
    
    /**
     * ربط القيم بالاستعلام
     */
    public function bind($param, $value, $type = null) {
        if (is_null($type)) {
            switch (true) {
                case is_int($value):
                    $type = PDO::PARAM_INT;
                    break;
                case is_bool($value):
                    $type = PDO::PARAM_BOOL;
                    break;
                case is_null($value):
                    $type = PDO::PARAM_NULL;
                    break;
                default:
                    $type = PDO::PARAM_STR;
            }
        }
        $this->stmt->bindValue($param, $value, $type);
        return $this;
    }
    
    /**
     * تنفيذ الاستعلام
     */
    public function execute() {
        return $this->stmt->execute();
    }
    
    /**
     * الحصول على صف واحد
     */
    public function fetch() {
        $this->execute();
        return $this->stmt->fetch();
    }
    
    /**
     * الحصول على جميع الصفوف
     */
    public function fetchAll() {
        $this->execute();
        return $this->stmt->fetchAll();
    }
    
    /**
     * الحصول على عدد الصفوف المتأثرة
     */
    public function rowCount() {
        return $this->stmt->rowCount();
    }
    
    /**
     * الحصول على آخر ID مُدرج
     */
    public function lastInsertId() {
        return $this->connection->lastInsertId();
    }
    
    /**
     * بدء معاملة
     */
    public function beginTransaction() {
        return $this->connection->beginTransaction();
    }
    
    /**
     * تأكيد المعاملة
     */
    public function commit() {
        return $this->connection->commit();
    }
    
    /**
     * إلغاء المعاملة
     */
    public function rollback() {
        return $this->connection->rollback();
    }
}

// فئة Car - للتعامل مع السيارات
class Car {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * الحصول على جميع السيارات المتاحة
     */
    public function getAvailableCars($limit = null) {
        $sql = "SELECT * FROM available_cars_view WHERE 1=1";
        if ($limit) {
            $sql .= " LIMIT :limit";
        }
        
        $this->db->prepare($sql);
        if ($limit) {
            $this->db->bind(':limit', $limit, PDO::PARAM_INT);
        }
        return $this->db->fetchAll();
    }
    
    /**
     * الحصول على سيارة بواسطة ID
     */
    public function getCarById($car_id) {
        $sql = "SELECT c.*, cat.category_name, 
                AVG(r.rating) as avg_rating,
                COUNT(r.review_id) as total_reviews
                FROM cars c
                LEFT JOIN car_categories cat ON c.category_id = cat.category_id
                LEFT JOIN reviews r ON c.car_id = r.car_id
                WHERE c.car_id = :car_id
                GROUP BY c.car_id";
        
        return $this->db->prepare($sql)
                       ->bind(':car_id', $car_id)
                       ->fetch();
    }
    
    /**
     * الحصول على ميزات السيارة
     */
    public function getCarFeatures($car_id) {
        $sql = "SELECT * FROM car_features WHERE car_id = :car_id";
        return $this->db->prepare($sql)
                       ->bind(':car_id', $car_id)
                       ->fetchAll();
    }
    
    /**
     * البحث عن السيارات
     */
    public function searchCars($keyword, $category = null) {
        $sql = "SELECT * FROM cars WHERE (car_name LIKE :keyword OR brand LIKE :keyword) 
                AND is_available = 1";
        
        if ($category) {
            $sql .= " AND category_id = :category";
        }
        
        $this->db->prepare($sql);
        $this->db->bind(':keyword', "%$keyword%");
        
        if ($category) {
            $this->db->bind(':category', $category);
        }
        
        return $this->db->fetchAll();
    }
}

// فئة Booking - للتعامل مع الحجوزات
class Booking {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * إنشاء حجز جديد
     */
    public function createBooking($data) {
        $sql = "INSERT INTO bookings (user_id, car_id, pickup_location, dropoff_location, 
                pickup_date, pickup_time, dropoff_date, dropoff_time, rental_type, 
                total_amount, final_amount) 
                VALUES (:user_id, :car_id, :pickup_location, :dropoff_location, 
                :pickup_date, :pickup_time, :dropoff_date, :dropoff_time, :rental_type, 
                :total_amount, :final_amount)";
        
        $this->db->prepare($sql);
        $this->db->bind(':user_id', $data['user_id']);
        $this->db->bind(':car_id', $data['car_id']);
        $this->db->bind(':pickup_location', $data['pickup_location']);
        $this->db->bind(':dropoff_location', $data['dropoff_location']);
        $this->db->bind(':pickup_date', $data['pickup_date']);
        $this->db->bind(':pickup_time', $data['pickup_time']);
        $this->db->bind(':dropoff_date', $data['dropoff_date']);
        $this->db->bind(':dropoff_time', $data['dropoff_time']);
        $this->db->bind(':rental_type', $data['rental_type']);
        $this->db->bind(':total_amount', $data['total_amount']);
        $this->db->bind(':final_amount', $data['final_amount']);
        
        if ($this->db->execute()) {
            return $this->db->lastInsertId();
        }
        return false;
    }
    
    /**
     * الحصول على حجوزات المستخدم
     */
    public function getUserBookings($user_id) {
        $sql = "SELECT b.*, c.car_name, c.brand, c.image_url 
                FROM bookings b
                JOIN cars c ON b.car_id = c.car_id
                WHERE b.user_id = :user_id
                ORDER BY b.created_at DESC";
        
        return $this->db->prepare($sql)
                       ->bind(':user_id', $user_id)
                       ->fetchAll();
    }
    
    /**
     * حساب تكلفة الحجز
     */
    public function calculateBookingCost($car_id, $rental_type, $pickup_date, $dropoff_date) {
        $sql = "CALL calculate_booking_cost(:car_id, :rental_type, :pickup_date, :dropoff_date, @total)";
        $this->db->prepare($sql);
        $this->db->bind(':car_id', $car_id);
        $this->db->bind(':rental_type', $rental_type);
        $this->db->bind(':pickup_date', $pickup_date);
        $this->db->bind(':dropoff_date', $dropoff_date);
        $this->db->execute();
        
        $result = $this->db->getConnection()->query("SELECT @total as total")->fetch();
        return $result['total'];
    }
}

// فئة User - للتعامل مع المستخدمين
class User {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * تسجيل مستخدم جديد
     */
    public function register($data) {
        $sql = "INSERT INTO users (full_name, email, phone, password_hash, city, country) 
                VALUES (:full_name, :email, :phone, :password, :city, :country)";
        
        $password_hash = password_hash($data['password'], PASSWORD_DEFAULT);
        
        $this->db->prepare($sql);
        $this->db->bind(':full_name', $data['full_name']);
        $this->db->bind(':email', $data['email']);
        $this->db->bind(':phone', $data['phone']);
        $this->db->bind(':password', $password_hash);
        $this->db->bind(':city', $data['city']);
        $this->db->bind(':country', $data['country']);
        
        if ($this->db->execute()) {
            return $this->db->lastInsertId();
        }
        return false;
    }
    
    /**
     * تسجيل الدخول
     */
    public function login($email, $password) {
        $sql = "SELECT * FROM users WHERE email = :email AND is_active = 1";
        $user = $this->db->prepare($sql)->bind(':email', $email)->fetch();
        
        if ($user && password_verify($password, $user['password_hash'])) {
            return $user;
        }
        return false;
    }
    
    /**
     * الحصول على معلومات المستخدم
     */
    public function getUserById($user_id) {
        $sql = "SELECT user_id, full_name, email, phone, address, city, country, user_type 
                FROM users WHERE user_id = :user_id";
        return $this->db->prepare($sql)->bind(':user_id', $user_id)->fetch();
    }
    
    /**
     * تحديث معلومات المستخدم
     */
    public function updateUser($user_id, $data) {
        $sql = "UPDATE users SET full_name = :full_name, phone = :phone, 
                address = :address, city = :city, country = :country 
                WHERE user_id = :user_id";
        
        $this->db->prepare($sql);
        $this->db->bind(':user_id', $user_id);
        $this->db->bind(':full_name', $data['full_name']);
        $this->db->bind(':phone', $data['phone']);
        $this->db->bind(':address', $data['address']);
        $this->db->bind(':city', $data['city']);
        $this->db->bind(':country', $data['country']);
        
        return $this->db->execute();
    }
}

// فئة Review - للتعامل مع التقييمات
class Review {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * إضافة تقييم جديد
     */
    public function addReview($data) {
        $sql = "INSERT INTO reviews (user_id, car_id, booking_id, rating, review_text) 
                VALUES (:user_id, :car_id, :booking_id, :rating, :review_text)";
        
        $this->db->prepare($sql);
        $this->db->bind(':user_id', $data['user_id']);
        $this->db->bind(':car_id', $data['car_id']);
        $this->db->bind(':booking_id', $data['booking_id']);
        $this->db->bind(':rating', $data['rating']);
        $this->db->bind(':review_text', $data['review_text']);
        
        return $this->db->execute();
    }
    
    /**
     * الحصول على تقييمات السيارة
     */
    public function getCarReviews($car_id, $approved_only = true) {
        $sql = "SELECT r.*, u.full_name 
                FROM reviews r
                JOIN users u ON r.user_id = u.user_id
                WHERE r.car_id = :car_id";
        
        if ($approved_only) {
            $sql .= " AND r.is_approved = 1";
        }
        
        $sql .= " ORDER BY r.created_at DESC";
        
        return $this->db->prepare($sql)->bind(':car_id', $car_id)->fetchAll();
    }
}

// فئة ContactMessage - لرسائل التواصل
class ContactMessage {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * إضافة رسالة تواصل
     */
    public function addMessage($data) {
        $sql = "INSERT INTO contact_messages (full_name, email, phone, subject, message) 
                VALUES (:full_name, :email, :phone, :subject, :message)";
        
        $this->db->prepare($sql);
        $this->db->bind(':full_name', $data['full_name']);
        $this->db->bind(':email', $data['email']);
        $this->db->bind(':phone', $data['phone']);
        $this->db->bind(':subject', $data['subject']);
        $this->db->bind(':message', $data['message']);
        
        return $this->db->execute();
    }
}

// فئة Blog - للمدونة
class Blog {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * الحصول على المقالات المنشورة
     */
    public function getPublishedPosts($limit = null) {
        $sql = "SELECT * FROM blog_posts 
                WHERE status = 'published' 
                ORDER BY published_at DESC";
        
        if ($limit) {
            $sql .= " LIMIT :limit";
        }
        
        $this->db->prepare($sql);
        if ($limit) {
            $this->db->bind(':limit', $limit, PDO::PARAM_INT);
        }
        
        return $this->db->fetchAll();
    }
    
    /**
     * الحصول على مقال بواسطة Slug
     */
    public function getPostBySlug($slug) {
        $sql = "SELECT p.*, u.full_name as author_name 
                FROM blog_posts p
                LEFT JOIN users u ON p.author_id = u.user_id
                WHERE p.slug = :slug AND p.status = 'published'";
        
        $post = $this->db->prepare($sql)->bind(':slug', $slug)->fetch();
        
        if ($post) {
            // زيادة عدد المشاهدات
            $this->incrementViews($post['post_id']);
        }
        
        return $post;
    }
    
    /**
     * زيادة عدد المشاهدات
     */
    private function incrementViews($post_id) {
        $sql = "UPDATE blog_posts SET views_count = views_count + 1 WHERE post_id = :post_id";
        $this->db->prepare($sql)->bind(':post_id', $post_id)->execute();
    }
}

// فئة Statistics - للإحصائيات
class Statistics {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * الحصول على إحصائيات الموقع
     */
    public function getSiteStats() {
        $sql = "SELECT * FROM site_statistics LIMIT 1";
        return $this->db->prepare($sql)->fetch();
    }
    
    /**
     * تحديث الإحصائيات
     */
    public function updateStats($data) {
        $sql = "UPDATE site_statistics SET 
                years_experience = :years_experience,
                total_cars = :total_cars,
                happy_customers = :happy_customers,
                total_branches = :total_branches";
        
        $this->db->prepare($sql);
        $this->db->bind(':years_experience', $data['years_experience']);
        $this->db->bind(':total_cars', $data['total_cars']);
        $this->db->bind(':happy_customers', $data['happy_customers']);
        $this->db->bind(':total_branches', $data['total_branches']);
        
        return $this->db->execute();
    }
}

// وظائف مساعدة عامة
class Helper {
    /**
     * تنظيف المدخلات
     */
    public static function cleanInput($data) {
        $data = trim($data);
        $data = stripslashes($data);
        $data = htmlspecialchars($data);
        return $data;
    }
    
    /**
     * التحقق من صحة البريد الإلكتروني
     */
    public static function validateEmail($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL);
    }
    
    /**
     * تنسيق التاريخ
     */
    public static function formatDate($date, $format = 'Y-m-d') {
        return date($format, strtotime($date));
    }
    
    /**
     * تنسيق المبلغ المالي
     */
    public static function formatPrice($amount, $currency = ') {
        return $currency . number_format($amount, 2);
    }
    
    /**
     * توليد رمز عشوائي
     */
    public static function generateToken($length = 32) {
        return bin2hex(random_bytes($length));
    }
    
    /**
     * إرسال بريد إلكتروني
     */
    public static function sendEmail($to, $subject, $message) {
        $headers = "From: " . ADMIN_EMAIL . "\r\n";
        $headers .= "Reply-To: " . ADMIN_EMAIL . "\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        
        return mail($to, $subject, $message, $headers);
    }
}

// معالجة الأخطاء
function handleError($errno, $errstr, $errfile, $errline) {
    $error_message = "Error [$errno]: $errstr in $errfile on line $errline";
    error_log($error_message);
    
    if (ini_get('display_errors')) {
        echo "<div style='background: #f8d7da; color: #721c24; padding: 10px; border: 1px solid #f5c6cb; border-radius: 4px; margin: 10px;'>";
        echo "<strong>خطأ:</strong> " . htmlspecialchars($errstr) . "<br>";
        echo "<strong>الملف:</strong> " . htmlspecialchars($errfile) . "<br>";
        echo "<strong>السطر:</strong> " . $errline;
        echo "</div>";
    }
}

set_error_handler("handleError");

// بدء الجلسة إذا لم تكن قد بدأت
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

?>